#!/bin/bash

# 检查是否提供了PID作为参数
if [ -z "$1" ]; then
    echo "错误：请提供要重启的进程的PID。"
    exit 1
fi

# 验证PID是否为数字
if ! [[ "$1" =~ ^[0-9]+$ ]]; then
    echo "错误：提供的PID必须是正整数。"
    exit 1
fi

PID=$1

# 检查进程是否存在
if ! ps -p "$PID" >/dev/null 2>&1; then
    echo "错误：未找到PID为 $PID 的进程。"
    exit 1
fi

# 获取进程详细信息
COMMAND_ARGS=()
CWD=""

# 优先使用/proc获取精确的命令行参数（Linux系统）
if [ -r "/proc/$PID/cmdline" ]; then
    # 读取cmdline并拆分为数组
    while IFS= read -r -d '' ARG; do
        COMMAND_ARGS+=("$ARG")
    done < "/proc/$PID/cmdline"

    # 获取工作目录
    CWD=$(readlink -f "/proc/$PID/cwd" 2>/dev/null)
else
    # 回退到ps命令（兼容其他Unix系统）
    echo "警告：无法访问/proc，尝试使用ps获取信息（可能不准确）。"
    COMMAND_STR=$(ps -p "$PID" -o args=)
    # 基本拆分（不处理复杂情况）
    read -ra COMMAND_ARGS <<< "$COMMAND_STR"
fi

# 验证命令参数
if [ ${#COMMAND_ARGS[@]} -eq 0 ]; then
    echo "错误：无法获取进程的命令行参数。"
    exit 1
fi

# 分离命令和参数
COMMAND="${COMMAND_ARGS[0]}"
ARGS=("${COMMAND_ARGS[@]:1}")

# 终止进程（先尝试SIGTERM，再强制SIGKILL）
if ! kill -15 "$PID" 2>/dev/null; then
    echo "警告：优雅终止失败，尝试强制终止..."
    kill -9 "$PID" 2>/dev/null || {
        echo "错误：无法终止进程 $PID。"
        exit 1
    }
fi

# 等待进程退出（最大5秒）
TIMEOUT=5
while ps -p "$PID" >/dev/null 2>&1 && [ $TIMEOUT -gt 0 ]; do
    sleep 1
    ((TIMEOUT--))
done

if [ $TIMEOUT -eq 0 ]; then
    echo "错误：进程 $PID 未正常退出。"
    exit 1
fi

# 切换工作目录（如果可用）
if [ -n "$CWD" ] && [ -d "$CWD" ]; then
    cd "$CWD" || echo "警告：无法切换到工作目录 $CWD"
fi

# 重启进程并获取新PID
nohup "${COMMAND}" "${ARGS[@]}" >/dev/null 2>&1 &
NEW_PID=$!

# 验证重启结果
sleep 1
if ps -p "$NEW_PID" >/dev/null 2>&1; then
    echo "进程已重启，新PID为 $NEW_PID。"
else
    echo "警告：进程重启可能失败，请手动验证。"
    exit 1
fi